#include "stdafx.h"
#include "RFApp.h"

#include "RFDoc.h"
#include "RFView.h"

#include "DSP.h"

#ifdef _DEBUG
#define new DEBUG_NEW
#endif

IMPLEMENT_DYNCREATE(CRFView, CView)

BEGIN_MESSAGE_MAP(CRFView, CView)
	// Standard printing commands
	ON_COMMAND(ID_FILE_PRINT, &CView::OnFilePrint)
	ON_COMMAND(ID_FILE_PRINT_DIRECT, &CView::OnFilePrint)
	ON_COMMAND(ID_FILE_PRINT_PREVIEW, &CRFView::OnFilePrintPreview)
	ON_WM_SIZE()
	ON_WM_CREATE()
	ON_WM_ERASEBKGND()
END_MESSAGE_MAP()

//
// Constructor
//
CRFView::CRFView()
{
	m_xLeftMarg = 40;
	m_xRightMarg = 40;
	m_yTopMarg = 140;
	m_yBotMarg = 40;

	pTrace = new std::vector<float>;
}

//
// Deconstructor
//
CRFView::~CRFView()
{
	wglMakeCurrent(m_cDC->GetSafeHdc(), m_hRC);

	glDeleteBuffers(1, &m_gratBuf);
	glDeleteBuffers(1, &m_traceBuf);
	glDeleteLists(m_fontBuf, 96);

	wglMakeCurrent(NULL, NULL);

	delete pTrace;
}

//
// PreCreateWindow
BOOL CRFView::PreCreateWindow(CREATESTRUCT& cs)
{
	cs.style |= (WS_CLIPCHILDREN | WS_CLIPSIBLINGS | CS_OWNDC);
	return CView::PreCreateWindow(cs);
}

//
// OnCreate Override
// Create OpenGL context
//
int CRFView::OnCreate(LPCREATESTRUCT lpCreateStruct)
{
	if (CView::OnCreate(lpCreateStruct) == -1)
		return -1;

	PIXELFORMATDESCRIPTOR pfd; 
    int pixelFormat; 

	m_cDC = new CClientDC( this );

	// Setup our pfd
	pfd.nSize = sizeof( PIXELFORMATDESCRIPTOR ); 
	pfd.nVersion = 1; 
	pfd.dwFlags = PFD_DRAW_TO_WINDOW | PFD_SUPPORT_OPENGL | PFD_DOUBLEBUFFER;
	pfd.iPixelType = PFD_TYPE_RGBA; 
	pfd.cColorBits = 24; 

	pixelFormat = ChoosePixelFormat(m_cDC->GetSafeHdc(), &pfd); 
	DescribePixelFormat(
		m_cDC->GetSafeHdc(), 
		pixelFormat, 				
		sizeof(PIXELFORMATDESCRIPTOR), 					
		&pfd); 

	if(SetPixelFormat(m_cDC->GetSafeHdc(), pixelFormat, &pfd) == FALSE) {
		// This should throw a messageBox	
		exit(1); 
	}

	m_hRC = wglCreateContext(m_cDC->GetSafeHdc()); 
	
	// One time gl Initialization stuff goes here
	GlInitialization(); 

	return 0;
}

void CRFView::GlInitialization()
{
	wglMakeCurrent(m_cDC->GetSafeHdc(), m_hRC);

	glewInit();
	glEnable(GL_DEPTH_TEST);
	glEnableClientState(GL_VERTEX_ARRAY);

	glDepthFunc(GL_LEQUAL);

	glShadeModel(GL_SMOOTH);
	glHint(GL_LINE_SMOOTH_HINT, GL_NICEST);

	glGenBuffers(1, &m_gratBuf);
	glGenBuffers(1, &m_traceBuf);

	// Create Graticule
	for(int i = 0; i <= 10; i++) { // Horz lines
		m_gratVec.push_back(0.f);
		m_gratVec.push_back(0.1f * i);
		m_gratVec.push_back(1.f);
		m_gratVec.push_back(0.1f * i);
	}
	for(int i = 0; i <= 10; i++) { // Vert lines
		m_gratVec.push_back(0.1f * i);
		m_gratVec.push_back(0.f);
		m_gratVec.push_back(0.1f * i);
		m_gratVec.push_back(1.f);
	}
	glBindBuffer(GL_ARRAY_BUFFER, m_gratBuf);
	glBufferData(GL_ARRAY_BUFFER, m_gratVec.size()*sizeof(float),
		&m_gratVec[0], GL_STATIC_DRAW);

	// Create Font
	m_fontBuf = glGenLists(96);
	m_fontHandle = GetFont(20, FW_NORMAL, "Arial");
	SelectObject(m_cDC->GetSafeHdc(), m_fontHandle);
	wglUseFontBitmaps(m_cDC->GetSafeHdc(), 32, 96, m_fontBuf);

	wglMakeCurrent(NULL, NULL);
}

//
// CRFView drawing
//
void CRFView::OnDraw(CDC* /*pDC*/)
{
	CRFDoc* pDoc = GetDocument();
	ASSERT_VALID(pDoc);
	if (!pDoc)
		return;

	wglMakeCurrent(m_cDC->GetSafeHdc(), m_hRC);

	// Get our dimensions
	m_xGrat = m_winWidth - (m_xRightMarg + m_xLeftMarg);
	m_yGrat = m_winHeight - (m_yTopMarg + m_yBotMarg);

	CreateNewTrace();

	glViewport(0, 0, m_winWidth, m_winHeight);
	glMatrixMode(GL_PROJECTION);
	glLoadIdentity();
	glOrtho(0, m_winWidth, 0, m_winHeight, -1, 1);

	glClearColor(35.f/255.f, 35.f/255.f, 35.f/255.f, 1.f);
	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

	// Drawing
	DrawGraticule();
	DrawSweep();
	DrawGratText();

	glFinish();
	SwapBuffers(m_cDC->GetSafeHdc());
	
	wglMakeCurrent(NULL, NULL);
}

//
// Get spectrum density (fft/etc.)
// Normalize 
//
void CRFView::CreateNewTrace()
{
	CRFDoc *pDoc = GetDocument();
	const bbSettings_t *pSet = pDoc->GetSettings();

	int sweepLen;
	const float *trace = pDoc->GetSweep(&sweepLen);

	NormalizeTrace(trace, REF_LEVEL, sweepLen);
}

//
// Normalize trace based on settings, add to our trace queue
//
void CRFView::NormalizeTrace(const float *t, double ref, int n)
{
	pTrace->clear();
	pTrace->reserve(n);
	
	float bot = ref - 200.0f;
	float xStep = 1.0f / n;
	float xScale = 1.0f / m_xGrat;
	float yScale = 1.0f / 200.0f;

	// Build the normalize trace
	for(int i = 0; i < n; i++) {

		float y = yScale * (t[i] - bot);
		y = min(y, 1.0f);
		y = max(y, 0.0f);

		pTrace->push_back(i * xStep);
		pTrace->push_back(y);
	}	
}

//
// Draw Graticule
//
void CRFView::DrawGraticule() 
{
	glColor3ub(255, 255, 255); //white

	glLineStipple(1, 0x8888);
	glEnable(GL_LINE_STIPPLE);

	glMatrixMode(GL_MODELVIEW);
	glLoadIdentity();

	glTranslatef(m_xLeftMarg, m_yBotMarg, 0.f);
	glScalef((float)m_xGrat, (float)m_yGrat, 1.f);

	glBindBuffer(GL_ARRAY_BUFFER, m_gratBuf);
	glVertexPointer(2, GL_FLOAT, 0, (GLvoid*)0);
	glDrawArrays(GL_LINES, 0, m_gratVec.size()/2);

	glDisable(GL_LINE_STIPPLE);
}

//
// Draw Grat Text
//
void CRFView::DrawGratText()
{
	CRFDoc *pDoc = GetDocument();
	const bbSettings_t *s = pDoc->GetSettings();
	const signalStats_t *stats = pDoc->GetStats();
	CString str;
	
	int sweepLen; float maxVal;
	const float *trace = pDoc->GetSweep(&sweepLen);
	ippsMax_32f(trace, sweepLen, &maxVal);

	glColor3ub(255, 255, 255);

	glMatrixMode(GL_MODELVIEW);
	glPushMatrix();
	glLoadIdentity();

	str.Format("Cen: %g MHz", s->center);
	RastText(str, LEFT_JUST, m_xLeftMarg+2, m_winHeight-m_yTopMarg-18);
	
	// Write bandwidth here
	str.Format("Span: %lf", 40.0e6 / (0x1 << s->decimation));
	RastText(str.GetBuffer(), LEFT_JUST, m_xLeftMarg+2, m_winHeight-m_yTopMarg-38);
	
	RastText("Ref: Full Scale", RIGHT_JUST, m_winWidth-m_xRightMarg-2, m_winHeight-m_yTopMarg-18);
	RastText("Div: 10dB", RIGHT_JUST, m_winWidth-m_xRightMarg-2, m_winHeight-m_yTopMarg-38);

	str.Format("Atten: %d dB", s->atten);
	RastText(str, CENTERED, m_winWidth/2, m_winHeight-m_yTopMarg-18);
	str.Format("Gain: %d", s->gain);
	RastText(str, CENTERED, m_winWidth/2, m_winHeight-m_yTopMarg-38);

	// Simple Stats
	RastText("Amplitudes from -32768 to +32764", LEFT_JUST, m_xLeftMarg+2, m_winHeight-38);
	str.Format("Max: %f", stats->maxAmp);
	RastText(str, LEFT_JUST, m_xLeftMarg+2, m_winHeight-78);
	str.Format("Avg Power: %f", stats->avgPower);
	RastText(str, LEFT_JUST, m_xLeftMarg+2, m_winHeight-98);

	if(pDoc->IsRecording()) {
		glColor3ub(255, 0, 0);
		RastText("Recording", RIGHT_JUST, m_winWidth-m_xRightMarg-2, m_winHeight-38);
	}

	// Overload text
	if(pDoc->IsOverload()) {
		glColor3ub(255, 0, 0);
		RastText("IF Overload", CENTERED, m_winWidth/2, m_winHeight-38);
	}

	glPopMatrix();
}

//
// Draw Trace in front of queue
//
void CRFView::DrawSweep()
{
	glPushAttrib(GL_VIEWPORT_BIT);
	glViewport(m_xLeftMarg, m_yBotMarg, m_xGrat, m_yGrat);

	// We do some nice drawing
	glColor3f(1.0, 1.0, 0.0);
	glEnable(GL_LINE_SMOOTH);
	glEnable(GL_BLEND);
	glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
	glBlendEquation(GL_FUNC_ADD);
	glLineWidth(1.0);

	glMatrixMode(GL_PROJECTION);
	glPushMatrix();
	glLoadIdentity();
	glOrtho(0, m_xGrat, 0, m_yGrat, 1, -1);

	glMatrixMode(GL_MODELVIEW);
	glPushMatrix();
	glLoadIdentity();
	glScalef(m_xGrat, m_yGrat, 1);

	glBindBuffer(GL_ARRAY_BUFFER, m_traceBuf);
	glBufferData(GL_ARRAY_BUFFER, 
		pTrace->size() * sizeof(float),
		&pTrace->front(),
		GL_DYNAMIC_DRAW);
	glVertexPointer(2, GL_FLOAT, 0, (GLvoid*)0);

	glDrawArrays(GL_LINE_STRIP, 0, pTrace->size() / 2);

	glMatrixMode(GL_PROJECTION);
	glPopMatrix();
	glMatrixMode(GL_MODELVIEW);
	glPopMatrix();

	// Disable anti-aliased lines
	glDisable(GL_LINE_SMOOTH);
	glDisable(GL_BLEND);
	glLineWidth(1.0);

	glPopAttrib();
}

//
// Get a custom font
//
HFONT CRFView::GetFont(int h, int w, char *name) 
{
	return CreateFont(
		h,                                // Height
		0,                                // Width
		0,                                // Angle of escapement
		0,                                // Orientation angle
		w,                                // Font Width
		FALSE,                            // Italic
		FALSE,                            // Underline
		FALSE,                            // Striked-out
		ANSI_CHARSET,                     // Charset
		OUT_TT_PRECIS,                    // Precision
		CLIP_DEFAULT_PRECIS,              // Clip precision
		ANTIALIASED_QUALITY,              // Output quality
		FF_DONTCARE | DEFAULT_PITCH,      // Family and Pitch
		name);                            // Font name
}

//
// Rasterize text
//
void CRFView::RastText(const CString &s, TEXT_JUST just, int x, int y)
{
	int xAdjusted;  // x value after adjusting for justification
	CRect r;

	switch(just) {
		case LEFT_JUST:
			xAdjusted = x;
			break;
		case RIGHT_JUST:
			DrawTextA(m_cDC->GetSafeHdc(), s, -1, r, DT_CALCRECT);
			//getTextRect(s, r);
			xAdjusted = x - r.Width();
			break;
		case CENTERED:
			DrawTextA(m_cDC->GetSafeHdc(), s, -1, r, DT_CALCRECT);
			//getTextRect(s, r);
			xAdjusted = x - r.Width() / 2;
			break;
		default: // Don't get here
			xAdjusted = x;
			break;
	}

	glRasterPos2i(xAdjusted, y);
	glPrint(s);
}

//
// Raster text
//
void CRFView::glPrint(const char *s)
{
	if(s == NULL) return;

	glPushAttrib(GL_LIST_BIT);
	glListBase(m_fontBuf - 32);
	glCallLists(strlen(s), GL_UNSIGNED_BYTE, s);
	glPopAttrib();
}

//
// View Printing
//
void CRFView::OnFilePrintPreview()
{
	AFXPrintPreview(this);
}

BOOL CRFView::OnPreparePrinting(CPrintInfo* pInfo)
{
	// default preparation
	return DoPreparePrinting(pInfo);
}

void CRFView::OnBeginPrinting(CDC* /*pDC*/, CPrintInfo* /*pInfo*/)
{
	// TODO: add extra initialization before printing
}

void CRFView::OnEndPrinting(CDC* /*pDC*/, CPrintInfo* /*pInfo*/)
{
	// TODO: add cleanup after printing
}

void CRFView::OnRButtonUp(UINT nFlags, CPoint point)
{
	ClientToScreen(&point);
	OnContextMenu(this, point);
}

void CRFView::OnContextMenu(CWnd* pWnd, CPoint point)
{
	theApp.GetContextMenuManager()->ShowPopupMenu(IDR_POPUP_EDIT, point.x, point.y, this, TRUE);
}

//
// Diagnostics
//
#ifdef _DEBUG
void CRFView::AssertValid() const
{
	CView::AssertValid();
}

void CRFView::Dump(CDumpContext& dc) const
{
	CView::Dump(dc);
}

CRFDoc* CRFView::GetDocument() const // non-debug version is inline
{
	ASSERT(m_pDocument->IsKindOf(RUNTIME_CLASS(CRFDoc)));
	return (CRFDoc*)m_pDocument;
}
#endif //_DEBUG

//
// View Resize
//
void CRFView::OnSize(UINT nType, int cx, int cy)
{
	CView::OnSize(nType, cx, cy);

	m_winWidth = cx;
	m_winHeight = cy;
}

//
// Prevent flickering
//
BOOL CRFView::OnEraseBkgnd(CDC* pDC)
{
	return true;
}
